import os
import shutil
import argparse
from email import policy
from email.parser import BytesParser
from email.utils import parseaddr
from datetime import datetime


def load_whitelist(whitelist_path):
    """
    whitelist 結構:
    {
        '王小明': {'ming@example.com'}
    }
    """
    whitelist = {}

    if not whitelist_path or not os.path.exists(whitelist_path):
        return whitelist

    with open(whitelist_path, 'r', encoding='utf-8') as f:
        for line in f:
            line = line.strip()
            if not line or '|' not in line:
                continue

            name, mail = line.split('|', 1)
            name = name.strip().lower()
            mail = mail.strip().lower()

            whitelist.setdefault(name, set()).add(mail)

    return whitelist


def write_log(log_dir, eml_path, reason):
    date_str = datetime.now().strftime("%Y%m%d")
    log_filename = f"{date_str}.log"
    log_path = os.path.join(log_dir, log_filename)

    time_str = datetime.now().strftime("%H:%M:%S")
    abs_eml_path = os.path.abspath(eml_path)

    with open(log_path, 'a', encoding='utf-8') as f:
        f.write(f"[{time_str}] 檔案: {abs_eml_path} | 原因: {reason}\n")


def process_eml(eml_path, target_dir, log_dir, whitelist):
    if not os.path.exists(eml_path):
        print(f"錯誤: 找不到 EML 檔案: {eml_path}")
        return

    try:
        with open(eml_path, 'rb') as f:
            msg = BytesParser(policy=policy.default).parse(f)
    except Exception as e:
        print(f"讀取 EML 失敗: {e}")
        return

    # ===== 解析寄件人 =====
    sender_name, sender_email = parseaddr(msg.get('from', ''))

    # 🔒 關鍵：清除 NULL 字元 + 正規化
    sender_name = sender_name.replace('\x00', '').strip().lower()
    sender_email = sender_email.strip().lower()

    # ===== 白名單沒有此姓名：直接略過 =====
    if sender_name not in whitelist:
        print(f"[略過] 白名單無此姓名：{sender_name}")
        return

    # ===== 白名單有此姓名，Email 正確：放行 =====
    if sender_email in whitelist[sender_name]:
        print(f"[放行] 白名單姓名 + Email 符合：{sender_name} <{sender_email}>")
        return

    # ===== 白名單有此姓名，但 Email 不符：冒名處理 =====
    print(f"[異常] 白名單姓名，但 Email 不符：{sender_name} <{sender_email}>")

    os.makedirs(target_dir, exist_ok=True)

    filename = os.path.basename(eml_path)
    dest_path = os.path.join(target_dir, filename)

    if os.path.exists(dest_path):
        import time
        base, ext = os.path.splitext(filename)
        dest_path = os.path.join(target_dir, f"{base}_{int(time.time())}{ext}")

    shutil.move(eml_path, dest_path)
    print(f"成功將 {filename} 移動至 {target_dir}")

    if log_dir:
        os.makedirs(log_dir, exist_ok=True)
        write_log(log_dir, eml_path, "符合白名單姓名，但 Email 不符")


def main():
    parser = argparse.ArgumentParser(
        description="EML 防冒名寄件檢查（姓名優先 + NULL 清理）"
    )
    parser.add_argument("eml_path", help="[必要] 要檢查的 .eml 檔案")
    parser.add_argument("target_dir", help="[必要] 冒名時移動目錄")
    parser.add_argument("log_dir", nargs='?', help="[可選] Log 目錄")
    parser.add_argument("whitelist_file", nargs='?', help="[可選] 寄件人白名單")

    args = parser.parse_args()

    whitelist = load_whitelist(args.whitelist_file)

    process_eml(
        args.eml_path,
        args.target_dir,
        args.log_dir,
        whitelist
    )


if __name__ == "__main__":
    main()
